#!/bin/bash
set -euo pipefail

# NewspaperDownloader macOS setup helper
# - Tries to clear macOS quarantine from the .app in this folder (if any)
# - Then launches the application

# 1) Ensure we are on macOS
if [ "$(uname)" != "Darwin" ]; then
  echo "This setup script is for macOS only."
  exit 1
fi

# 2) Get script directory
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

# 3) Locate the .app (default name, or auto-detect if not found)
APP_NAME="NewspaperDownloader.app"
APP_PATH="${SCRIPT_DIR}/${APP_NAME}"

if [ ! -d "$APP_PATH" ]; then
  # auto-detect a single .app in this folder
  shopt -s nullglob
  app_candidates=("${SCRIPT_DIR}"/*.app)
  shopt -u nullglob

  if [ "${#app_candidates[@]}" -eq 1 ] && [ -d "${app_candidates[0]}" ]; then
    APP_PATH="${app_candidates[0]}"
    APP_NAME="$(basename "$APP_PATH")"
  else
    osascript -e 'display dialog "Could not find \"NewspaperDownloader.app\" in this folder.\n\nMake sure Setup.command is in the same folder as the .app bundle." buttons {"OK"} default button 1'
    exit 1
  fi
fi

# 4) Try to remove quarantine (ignore all errors, OK if there is no attribute)
xattr -dr com.apple.quarantine "$APP_PATH" 2>/dev/null || true

# 5) Always show success message
osascript -e 'display dialog "已完成设置，正在启动..." buttons {"OK"} default button 1 giving up after 2'

# 6) Start the app
if ! open "$APP_PATH"; then
  osascript -e "display dialog \"Failed to start \\\"${APP_NAME}\\\".\n\nPlease try opening it manually from Finder.\" buttons {\"OK\"} default button 1"
  exit 1
fi

exit0

